/*==============================================================================
SCATTER PLOT BANK VALUES WITH UNINSURED CAP
==============================================================================
Purpose:
This script generates a specialized scatter plot for Extension 6 where uninsured 
deposits are capped. It shows bank value components against uninsured share with
a purple ribbon indicating the cap region and automatic bank annotations.

Input:
- Analysis-ready dataset with uninsured cap extension data
- Global variable: $uninsured_cap (the cap level, e.g., 0.6)
- Global variable: $fig_scatter_annotated_ext (output filename)

Output:
- Figure A10 showing the uninsured cap effect

Paper Correspondence:
- Extension 6: Figure A.10

Last updated: July 03, 2025
==============================================================================*/

display "--- Starting Scatter Plot Bank Values with Uninsured Cap ---"

**===============================================================================
* Step 1: Setup and Data Preparation
*===============================================================================

* Define RSSD IDs for the specific banks to be highlighted
local svb_rssdid = 802866 // Silicon Valley Bank
local sig_rssdid = 2942690 // Signature Bank (note: different ID for this extension)
local frb_rssdid = 4114567 // First Republic Bank (note: different ID for this extension)
local highlight_rssdids = "`svb_rssdid', `sig_rssdid', `frb_rssdid'"

* Get the uninsured cap level from global variable
local uninsured_cap = ${uninsured_cap}
if "`uninsured_cap'" == "" {
    local uninsured_cap = 0.6  // Default cap level
}

* Define small shift for bank positioning to avoid overlap
local shiftby = 0.0125

* Create temporary variable for plotting (preserve original data)
gen uninsuredsh_domdep_plot = uninsuredsh_domdep

* Apply small shifts to highlighted banks to avoid visual overlap
replace uninsuredsh_domdep_plot = uninsuredsh_domdep_plot - `shiftby' if rssdid==`frb_rssdid'
replace uninsuredsh_domdep_plot = uninsuredsh_domdep_plot + `shiftby' if rssdid==`svb_rssdid'
* Signature bank keeps original position

**===============================================================================
* Step 2: Calculate Bank Value Ranges and Box Coordinates
*===============================================================================

* Use Feb 2023 period data (extension focus)
local period_filter = "feb2023"
local yvar_noDF = "v_noDF"       // Bank value (no deposit franchise) - long form
local yvar_0 = "v_0"             // Bank value (insured deposits only) - long form  
local yvar_1 = "v_1"             // Bank value (all deposits) - long form

* Calculate min/max values for each highlighted bank for box drawing
gen v_max_temp = max(`yvar_noDF', `yvar_0', `yvar_1') if period == "`period_filter'"
gen v_min_temp = min(`yvar_noDF', `yvar_0', `yvar_1') if period == "`period_filter'"

* Store box coordinates for each bank
local box_width_half = 0.005
local box_height_buffer = 0.8
local lcolorstyle = "black%50"

* Initialize locals for box lines and labels
local box_lines = ""
local bank_labels = ""

* Loop through highlighted banks to calculate positions
foreach rssdid of numlist `highlight_rssdids' {
    
    * Get bank name
    local bank_name = ""
    if `rssdid' == `svb_rssdid' local bank_name = "SVB"
    if `rssdid' == `sig_rssdid' local bank_name = "Signature" 
    if `rssdid' == `frb_rssdid' local bank_name = "First Republic"
    
    * Get coordinates for this bank
    sum uninsuredsh_domdep_plot if rssdid==`rssdid' & period=="`period_filter'", meanonly
    if r(N) > 0 {
        local x_coord = r(mean)
        
        sum v_max_temp if rssdid==`rssdid' & period=="`period_filter'", meanonly
        local y_max = r(mean) + `box_height_buffer'
        
        sum v_min_temp if rssdid==`rssdid' & period=="`period_filter'", meanonly  
        local y_min = r(mean) - `box_height_buffer'
        
        * Calculate box coordinates
        local x_left = `x_coord' - `box_width_half'
        local x_right = `x_coord' + `box_width_half'
        
        * Add box lines
        local box_lines "`box_lines' (scatteri `y_max' `x_left' `y_max' `x_right', recast(line) lc(`lcolorstyle'))"
        local box_lines "`box_lines' (scatteri `y_min' `x_left' `y_min' `x_right', recast(line) lc(`lcolorstyle'))"
        local box_lines "`box_lines' (scatteri `y_max' `x_left' `y_min' `x_left', recast(line) lc(`lcolorstyle'))"
        local box_lines "`box_lines' (scatteri `y_max' `x_right' `y_min' `x_right', recast(line) lc(`lcolorstyle'))"
        
        * Determine label position (above or below box)
        local label_y = `y_max' + 0.8
        local label_placement = "north"
        
        * Special positioning for Signature (place below)
        if `rssdid' == `sig_rssdid' {
            local label_y = `y_min' - 0.8
            local label_placement = "south"
        }
        
        * Add label
        local bank_labels "`bank_labels' text(`label_y' `x_coord' "`bank_name'", color(black) orient(vertical) placement(`label_placement'))"
    }
}

* Clean up temporary variables
drop v_max_temp v_min_temp

**===============================================================================
* Step 3: Define Purple Ribbon for Cap Region
*===============================================================================

* Calculate ribbon boundaries around the cap
local ribbon_left = `uninsured_cap' - 0.02
local ribbon_right = `uninsured_cap' + 0.02

* Purple ribbon showing the cap region
local purple_ribbon = "(function y=25, range(`ribbon_left' `ribbon_right') recast(area) color(purple%20) lw(none) base(-10) yaxis(1))"

**===============================================================================
* Step 4: Generate the Scatter Plot
*===============================================================================

* Create the comprehensive scatter plot
twoway `purple_ribbon' /// Purple ribbon for cap region
    `box_lines' /// Bank annotation boxes
    (scatter `yvar_1' uninsuredsh_domdep_plot if period=="`period_filter'" & assets_bill>=100, ///
        msize(2) mcolor(sand%30) lcolor(sand%30)) /// All banks - full value
    (scatter `yvar_0' uninsuredsh_domdep_plot if period=="`period_filter'" & assets_bill>=100, ///
        msize(2) mcolor(midblue%30) lcolor(midblue%30)) /// All banks - run value  
    (scatter `yvar_noDF' uninsuredsh_domdep_plot if period=="`period_filter'" & assets_bill>=100, ///
        msize(2) mcolor(red%30) lcolor(red%30) xtitle("Uninsured %") ytitle("Bank value")) /// All banks - no DF
    (scatter `yvar_1' uninsuredsh_domdep_plot if period=="`period_filter'" & inlist(rssdid,`highlight_rssdids'), ///
        msize(2) mcolor(sand) lcolor(sand)) /// Highlighted banks - full value
    (scatter `yvar_0' uninsuredsh_domdep_plot if period=="`period_filter'" & inlist(rssdid,`highlight_rssdids'), ///
        msize(2) mcolor(midblue) lcolor(midblue)) /// Highlighted banks - run value
    (scatter `yvar_noDF' uninsuredsh_domdep_plot if period=="`period_filter'" & inlist(rssdid,`highlight_rssdids'), ///
        msize(2) mcolor(red) lcolor(red)), /// Highlighted banks - no DF
    /* Graph styling and options */ ///
    legend(order(19 "A {&minus} D " 18 "A {&minus} D + DF{subscript:I}" 17 "A {&minus} D + DF{subscript:I} + DF{subscript:U}") ///
           pos(10) col(1) ring(0)) /// Legend positioning
    plotregion(margin(0 0 0 0)) /// Plot margins
    xlabel(0.2(0.2)`=`uninsured_cap'+0.05') /// X-axis labels extending slightly past cap
    graphregion(color(white)) /// Background color
    ylabel(-10(5)25, grid) /// Y-axis labels with grid
    yscale(r(-5 25)) /// Y-axis range (capped appropriately)
    xsize(7) /// Graph width
    scheme(s1color) /// Color scheme
    text(23 `uninsured_cap' "Uninsured %" "capped at `uninsured_cap'", color(purple)) /// Cap annotation
    `bank_labels' // Bank name labels

**===============================================================================
* Step 5: Export Figure
*===============================================================================

* Export using the global filename variable
graph export "$path_figures/${fig_scatter_annotated_ext}.pdf", as(pdf) fontface(Georgia) replace

* Clean up temporary variable
drop uninsuredsh_domdep_plot

display "--- Scatter Plot Bank Values with Uninsured Cap completed ---"
display "--- Output saved as: ${fig_scatter_annotated_ext}.pdf ---"
